import type { NextApiRequest, NextApiResponse } from 'next';
import OpenAI from 'openai';
import { prisma } from '@/lib/prisma';

export default async function handler(
    req: NextApiRequest,
    res: NextApiResponse
) {
    if (req.method !== 'POST') {
        res.setHeader('Allow', ['POST']);
        return res.status(405).end(`Method ${req.method} Not Allowed`);
    }

    const { clientId } = req.body;

    if (!clientId) {
        return res.status(400).json({ error: 'Missing clientId' });
    }

    try {
        const client = await prisma.client.findUnique({
            where: { id: clientId },
        });

        if (!client) {
            return res.status(404).json({ error: 'Client not found' });
        }

        const openai = new OpenAI({
            apiKey: client.openaiApiKey,
        });

        // Create a ChatKit session
        // Note: The exact method signature depends on the ChatKit SDK version.
        // Assuming openai.chatkit.sessions.create is available as per README.
        // If not, we might need to check the SDK docs or types.
        // Based on README: session = openai.chatkit.sessions.create({...})

        // We can pass the workflowId if ChatKit supports it in the session creation
        // or if it's used elsewhere. The prompt mentioned "Workflow ID".
        // If ChatKit sessions don't take a workflow ID, it might be part of the model or other config.
        // For now, I'll assume we just need to create a session.

        // @ts-ignore - ChatKit types might not be fully exposed in the openai package yet
        const session = await openai.chatkit.sessions.create({
            // You might need to pass specific parameters here depending on your setup
            // For example, model, or specific capabilities.
            // For now, we'll keep it minimal.
        });

        return res.status(200).json({ client_secret: session.client_secret });
    } catch (error: any) {
        console.error('Error creating session:', error);
        return res.status(500).json({ error: 'Failed to create session', details: error.message });
    }
}
